/*************************************************************************************************
 *
 *   Copyright (c) Hilscher GmbH. All Rights Reserved.
 *
 *************************************************************************************************/
/**
 * @file Application.c
 *
 * @brief This file contains some common functions which might be used by any example application. 
 * 
 * @author R. Walter (basis: E. Ott)
 *
 */


#include "Debug.h"
#include "Common.h"
#include "Application.h"

#ifdef WIN32
  #include "conio.h"
#endif


/**
 * @brief Common helper function for sending requests and receiving confirmations.
 * Unknown packets will be forwarded to the common packet handler.
 * This function may only be used during system startup, 
 * when no indications are expected to come up. 
 * Cascading calls of this function may end up in trouble.
 *
 * @param ptApp        Pointer to application data.
 * @param ptSendPkt    Pointer to packet which should be send (Request).
 * @param ptRecvPkt    Pointer to received packet (Confirmation).
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_SendRecvPkt(APPLICATION_T* ptApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt)
{
  /* return value of function */	
  TLR_RESULT tResult = TLR_S_OK; 

  /* the ulCnfCmd is always: ulReqCmd | 0x01 */
  TLR_UINT32 ulCnfCmd = ptSendPkt->tHeader.ulCmd | 0x01;  
  
  /* fire the packet */
  tResult = xChannelPutPacket(ptApp->tCommon.hChannel, ptSendPkt, ptApp->tCommon.ulTimeout);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* ok, at this point we have successfully sent a packet */

  while(CIFX_NO_ERROR == (tResult = xChannelGetPacket(ptApp->tCommon.hChannel, sizeof(*ptRecvPkt), ptRecvPkt, ptApp->tCommon.ulTimeout)))
  {
    /* check for our packet */
    if(ptRecvPkt->tHeader.ulCmd == ulCnfCmd)
    {
      /* it is our packet, so return its status as result */
      tResult = ptRecvPkt->tHeader.ulState;
      
      /* Note: we also return the packet which we have received (by reference, see signature of function) */

      /* we have received our packet, so we can break here */
      break;
    }
    else
    {
      /* it is something else, so place it in the application packet handler */
      App_HandlePacket(ptApp, ptRecvPkt);
    }
  }

  return tResult;
}


/**
 * @brief Common helper function for sending empty requests (only command codes) 
 * and receiving confirmations. This packet may be used for any registration effords.
 * Unknown packets will be forwarded to the common packet handler.
 * This function may only be used during system startup, 
 * when no indications are expected to come up. 
 * Cascading calls of this function may end up in trouble.
 *
 * @param ptApp        Pointer to application data.
 * @param ulCmd        Command code which should be used for empty packet.
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_SendRecvEmptyPkt(APPLICATION_T* ptApp, TLR_UINT32 ulCmd)
{
  /* return value of function */	
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};


  /* all register packets are empty packet, */
  /* which means that they contain no data, */
  /* so a cast to a specific type is not necessary */
  TLR_EMPTY_PACKET_T* ptReq = (TLR_EMPTY_PACKET_T*) &tSendPkt;

  /* ulSrc and ulDest are set by GetPacket / SendPacket */
  ptReq->tHead.ulCmd    = ulCmd;
  ptReq->tHead.ulDest   = 0x20;
  ptReq->tHead.ulDestId = 0;
  ptReq->tHead.ulExt    = 0;
  ptReq->tHead.ulId     = 0;
  ptReq->tHead.ulLen    = 0;
  ptReq->tHead.ulRout   = 0;
  ptReq->tHead.ulSrc    = 0;
  ptReq->tHead.ulSrcId  = 0;
  ptReq->tHead.ulSta    = 0;

  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }
  
  return tResult;
}



/**
 * @brief Main entry function which manages the startup of the application.
 *
 * @param ptApp        Pointer to application data.
 * 
 * @return TLR_RESULT  Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Main(APPLICATION_T* ptApp)
{
  /* return value of main function */
  TLR_RESULT tResult = TLR_S_OK;
  
  /* name of board to use */
  char* szBoardName = "CIFx0";

  /* number of channel to use */
  TLR_UINT32 ulChannel = 0;

  /* general reference to driver */
  CIFXHANDLE  hDriver  = NULL;

  /* general reference to communication channel */
  CIFXHANDLE  hChannel = NULL;

  /* common variables for packets */
  CIFX_PACKET tSendPkt  = {{0}};
  CIFX_PACKET tRecvPkt  = {{0}};
  CIFX_PACKET* ptRecvPkt = NULL;

  /* variable for host state */
  TLR_UINT32  ulState     = 0;

  /* default timeout for driver api calls */
  TLR_UINT32  ulTimeout   = 100;        

  /* prompt a startup screen when running in debug mode */
  tResult = App_PromptIntro(ptApp);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* place default timeout value in resources */
  ptApp->tCommon.ulTimeout = ulTimeout;

  /* Open Driver */
  DEBUG("Opening driver...\n");
  tResult = xDriverOpen(&hDriver);
  
  if (CIFX_NO_ERROR == tResult)
  {
    /* place driver handle in application resources */
    ptApp->tCommon.hDriver = hDriver;

    /* Driver successfully opened */
    /* Open channel */
    DEBUG("Opening channel %d on board %s...\n", ulChannel, szBoardName);
    tResult = xChannelOpen(hDriver, szBoardName, ulChannel, &hChannel);

    if (CIFX_NO_ERROR == tResult)
    {
      /* place channel handle in application resources */
      ptApp->tCommon.hChannel = hChannel;

      /* start hardware with new configuration */
      DEBUG("Processing system restart...\n");

#ifdef WIN32      
      tResult = xChannelReset(hChannel, CIFX_SYSTEMSTART, 2000); 
#endif
      
      /* system restart successful */
      if (CIFX_NO_ERROR == tResult)
      {
        /* Toggle Application Ready State Flag */
        do
        {
          tResult = xChannelHostState(hChannel, CIFX_HOST_STATE_READY, &ulState, ulTimeout);
          
          /* if Dev is not ready, retry that action */
          if (CIFX_DEV_NOT_READY == tResult)
          {
#ifdef WIN32
            /* retry after 500 ms */
            Sleep(500);
#endif
          }
        } 
        while (tResult == CIFX_DEV_NOT_READY);      


        /* check for CIFX_NO_ERROR added because return value  of xChannelHostState() changed */
        /* between SHM-lib <=0.930 and >=0.940 */
        if ((CIFX_NO_ERROR == tResult) || (CIFX_DEV_NOT_RUNNING == tResult))
        {
          /* build configuration packet
           * 
           * We use V4 configuration service here, 
           * but V2 configuration packet could be used as well (legacy).
           */
          /* tResult = App_BuildConfigurationReqV2(&tSendPkt); */
          tResult = App_BuildConfigurationReqV4(&tSendPkt);

          /* send configuration Req packet */
          DEBUG("Sending configuration request...\n");
          tResult = xChannelPutPacket(hChannel, &tSendPkt, ulTimeout);

          /* get configuration Cnf packet */
          tResult = xChannelGetPacket(hChannel, sizeof(tRecvPkt), &tRecvPkt, ulTimeout);

          /* check if we got an error within configuration packet */
          if (TLR_S_OK != tResult)
          {
            DEBUG("Configuration Confirmation not received.");
          }
          else if (TLR_S_OK != tRecvPkt.tHeader.ulState)
          {
            DEBUG("Configuration packet returned with error code: 0x%x\n", tRecvPkt.tHeader.ulState );
          }
          else
          {
            /* start hardware with new configuration */
            DEBUG("Processing channel init...\n");
            tResult = xChannelReset(hChannel, CIFX_CHANNELINIT, 2000);
          
            if (CIFX_NO_ERROR == tResult)
            {
              /* Waiting for netX warmstarting */
              do
              {
                tResult = xChannelHostState(hChannel, CIFX_HOST_STATE_READY, &ulState, ulTimeout);
              }
              while (CIFX_DEV_NOT_RUNNING == tResult);
            
            
              /* check CifX state */
              if ((CIFX_NO_ERROR == tResult) || (CIFX_DEV_NO_COM_FLAG == tResult))
              {
                /* initialize the application */
                tResult = App_Initialize(ptApp);

                /* check if initializing the application succeeded */
                if ((TLR_S_OK == tResult) && (NULL != ptApp))
                {
                  /* bus on */
                  DEBUG("Setting bus state on...\n");
                  tResult = xChannelBusState(hChannel, CIFX_BUS_STATE_ON, &ulState, ulTimeout);        
          
                  /* bus activity begins here */
                  if (CIFX_BUS_STATE_ON == ulState)
                  {
                    DEBUG("Entering endless loop...\n");
                    DEBUG("\n");
                    DEBUG(">> Press any key to leave. <<\n");
                    DEBUG("\n");
          
                    /* set return to TLR_S_OK to initially enter the loop */
                    tResult = TLR_S_OK;
                    while(TLR_S_OK == tResult || CIFX_DEV_GET_NO_PACKET == tResult)
                    {
                      /* do process stuff first */
                      /* handling of io depends on the bus system,  */
                      /* there might be some bits we have to toggle */
                      App_HandleProcessData(ptApp);                   
            
                      /* now we do the acyclic stuff*/
                      tResult = xChannelGetPacket(hChannel, sizeof(tRecvPkt), &tRecvPkt, 5);
            
                      /* check if we have really received a packet or an error code */
                      if(TLR_S_OK == tResult)
                      {
                        /* signalize if we got a packet*/
                        DEBUG("-");
            
                        /* handle packets */
                        App_HandlePacket(ptApp, &tRecvPkt);
                      }

#ifdef WIN32
                      /* enable leaving of endless loop if running under windows, */
                      /* necessary to return handles and close driver */
                      if (_kbhit())
                      {
                        tResult = TLR_E_FAIL;
                      }
#endif
                          
                    } /* while(TLR_S_OK == tResult || CIFX_DEV_GET_NO_PACKET == tResult) --> packet loop */
  
                    /* force at least one empty line within console output */
                    DEBUG("\n\n");
  
                  } /* Bus on */
                
                  /* Finalize application in order to dispose handles, free memory, aso.*/
                  App_Finalize(ptApp);
  
                } /* check initialize application succeeded */

              } /* check CifX state */

            } /* cifX channel init */

          } /* check configuration packet */

        } /* check for CIFX_NO_ERROR added because return value of xChannelHostState() changed */

      } /* cifX successfully reset */
      
      DEBUG("Closing Channel...\n");
      tResult = xChannelClose(hChannel);
    } /* channel successfully opened */

    DEBUG("Closing Driver...\n");
    tResult = xDriverClose(hDriver);

  } /* driver successfully opened */

  /* bye, bye, ...*/
  return tResult;
}
